<?php
namespace app\service\business;

use app\common\exception\CommonException;
use app\common\util\CommonUtils;
use app\common\util\PlatformUtils;
use app\common\util\RequestUtils;
use app\service\CoinService;
use app\service\CollectService;
use app\service\MemberService;
use app\service\PointService;
use app\service\SessionService;
use app\service\ShopService;
use think\facade\Db;

class MemberBindMobileService
{
    protected $shopService;
    protected $collectService;
    protected $memberService;
    protected $coinService;
    protected $pointService;
    protected $requestUtils;
    protected $sessionService;
    protected $platformUtils;

    // 构造函数用于依赖注入
    public function __construct(
        ShopService $shopService,
        CollectService $collectService,
        MemberService $memberService,
        CoinService $coinService,
        PointService $pointService,
        PlatformUtils $platformUtils,
        SessionService $sessionService,
        RequestUtils $requestUtils
    ) {
        $this->shopService = $shopService;
        $this->collectService = $collectService;
        $this->memberService = $memberService;
        $this->coinService = $coinService;
        $this->pointService = $pointService;
        $this->requestUtils = $requestUtils;
        $this->platformUtils = $platformUtils;
        $this->sessionService = $sessionService;
    }
    
    protected $mobile = '';
    protected $collect = [];
    protected $mem = [];
    protected $mobileMem = [];
    protected $mobileCollect = [];
    protected $setSessionInfo = [];

    // 涉及session操作
    public function handle($mobile, $collect_id, $mem_id, $shop_id)
    {
        $this->mobile = $mobile;
        $this->checkCurrent($collect_id, $mem_id, $shop_id);
        $this->checkMobile();

        // 启动事务
        Db::startTrans();
        try {
            if ($this->mobileMem) {
                if ($this->mobileCollect) {
                    $return = $this->handleByBindMobileToShop();
                } else {
                    $return = $this->handleByBindMobileToNotShop();
                }
            } else {
                // 没绑定手机号的直接绑定
                $return = $this->handleByNotBindMobile();
            }
            // 提交事务
            Db::commit();
        } catch (CommonException $e) {
            // 回滚事务
            Db::rollback();
            throw new CommonException($e->getMessage(), $e->getCode(), $e->getData());
        } catch (\Exception $e) {
            // 回滚事务
            Db::rollback();
            throw new CommonException($e->getMessage());
        }
        // 修改当前session信息
        if (isset($this->setSessionInfo['mem_id'])) {
            $this->sessionService->registerSessionByMemberLogin(
                $this->setSessionInfo['mem_id'],
                $this->setSessionInfo['shop_id'],
                $this->setSessionInfo['collect_id']
            );
        }
        return $return;
    }
    
    protected function checkCurrent($collect_id, $mem_id, $shop_id)
    {
        // 查询当前关联关系
        $collect = $this->collectService->getByID($collect_id);
        if (!$collect) {
            throw new CommonException('账号关系不存在');
        }
        if ($collect['mem_id'] != $mem_id || $collect['shop_id'] != $shop_id) {
            throw new CommonException('账号关系错误');
        }
        $mem = $this->memberService->getByMemID($collect['mem_id']);
        if (!$mem) {
            throw new CommonException('登录账号信息不存在');
        }
        if ($mem['mem_hand']) {
            throw new CommonException('当前账号已绑定手机号');
        }
        $this->collect = $collect;
        $this->mem = $mem;
    }
    protected function checkMobile()
    {
        $mobile = $this->mobile;
        if (!$mobile) {
            throw new CommonException('请输入手机号');
        }
        // 查询手机号绑定
        $mobileMem = $this->memberService->getByMobile($mobile);
        $this->mobileMem = $mobileMem;
        if ($mobileMem) {
            // 查询手机号账号关系
            $mobileCollect = $this->collectService->getByMemIDAndShopID($mobileMem['mem_id'], $this->collect['shop_id']);
            $this->mobileCollect = $mobileCollect;
        } else {
            $this->mobileCollect = [];
        }
    }

    protected function handleByNotBindMobile()
    {
        $up = array(
            'mem_hand'=> $this->mobile
        );
        Db::table('msm_member')->where('mem_id', $this->mem['mem_id'])->update($up);
        $return = [
            'mem_id' => $this->mem['mem_id'] ?? 0,
        ];
        return $return;
    }
    
    protected function handleByBindMobileToShop()
    {
        // 该手机号已绑定过该店铺，检测有无绑定抖音账号
        if ($this->mobileCollect['dymp_openid']) {
            throw new CommonException('该手机号已绑定抖音账号，请勿重复绑定');
        }
        // 手机号对应的关注绑定抖音
        $up = array(
            'dymp_openid'=> $this->collect['dymp_openid'],
            'dymp_unionId'=> $this->collect['dymp_unionId'],
        );
        $up_id = Db::table('msm_member_collect')->where('id', $this->mobileCollect['id'])->update($up);
        // 当前关注解除
        $up = array(
            'dymp_openid'=> '',
            'dymp_unionId'=> '',
            'unbind_dymp_openid'=> $this->collect['dymp_openid'],
            'unbind_dymp_unionId'=> $this->collect['dymp_unionId'],
        );
        $up_id = Db::table('msm_member_collect')->where('id', $this->collect['id'])->update($up);
        $this->setSessionInfo = [
            'mem_id' => $this->mobileCollect['mem_id'],
            'shop_id' => $this->mobileCollect['shop_id'],
            'collect_id' => $this->mobileCollect['id'],
        ];
        // 积分以及豆子的转移逻辑
        $this->changeCoinAndPoint($this->collect, $this->mobileCollect);
        $newMemID = $this->mobileCollect['mem_id'];
        $return = [
            'mem_id' => $newMemID,
        ];
        return $return;
    }
    protected function handleByBindMobileToNotShop()
    {
        // 该手机号未绑定过该店铺，直接更换当前关联关系的mem_id
        // 将关系的mem_id 和 unbind_mem_id 记录修改
        $up = array(
            'mem_id'=> $this->mobileMem['mem_id'],
            'unbind_mem_id'=> $this->collect['mem_id'],
        );
        $up_id = Db::table('msm_member_collect')->where('id', $this->collect['id'])->update($up);
        $this->setSessionInfo = [
            'mem_id' => $this->mobileMem['mem_id'],
            'shop_id' => $this->collect['shop_id'],
            'collect_id' => $this->collect['id'],
        ];
        // 重新请求
        $newCollect = $this->collectService->getByID($this->collect['id']);
        if (!$newCollect) {
            $newCollect = $this->collect;
        }
        // 积分以及豆子的转移逻辑
        $this->changeCoinAndPoint($this->collect, $newCollect);
        $newMemID = $this->mobileMem['mem_id'];
        $return = [
            'mem_id' => $newMemID,
        ];
        return $return;
    }

    protected function changeCoinAndPoint($outCollect, $inCollect)
    {
        // 豆子的转移逻辑
        $outFee = $this->memberService->getFeeByMemID($outCollect['mem_id']);
        $outCoin = (int)($outFee['coin'] ?? 0);
        $outCoinUsed = (int)($outFee['coin_used'] ?? 0);
        $outCoinUse = (int)($outCoin - $outCoinUsed);
        $outPoint = (string)($outFee['fee_leave'] ?? '0.00');
        if ($outCoinUse <= 0 && $outPoint <= 0) {
            return [];
        }
        $inFee = $this->memberService->getFeeByMemID($inCollect['mem_id']);
        $inCoin = (int)($inFee['coin'] ?? 0);
        $inCoinUsed = (int)($inFee['coin_used'] ?? 0);
        $inCoinUse = (int)($inCoin - $inCoinUsed);
        $inPoint = (string)($inFee['fee_leave'] ?? '0.00');
        $inFeeMoney = (string)($inFee['fee_money'] ?? '0.00');
        if (!$inFee) {
            $feeData = [
                'mem_id' => $inCollect['mem_id'],
                'shop_id' => $inCollect['shop_id'],
            ];
            $fee_id = Db::table('msm_member_fee')->insertGetId($feeData);
            if (!$fee_id) {
                throw new CommonException('会员信息创建失败');
            }
        }
        $order_no = CommonUtils::getByAdOrderNo($this->platformUtils->getPlatform());
        if ($outCoinUse > 0) {
            // 转出豆子
            $coinData = array(
                'category' => CoinService::CATEGORY_REDUCE_MOBILE,
                'out_collect_id' => $outCollect['id'],
                'out_mem_id' => $outCollect['mem_id'],
                'out_shop_id' => $outCollect['shop_id'],
                'in_collect_id' => $inCollect['id'],
                'in_mem_id' => $inCollect['mem_id'],
                'in_shop_id' => $inCollect['shop_id'],
                'coin' => -$outCoinUse,
                'old_coin' => $outCoinUse,
                'order_no' => $order_no,
                'qrcode_code' => '',
                'remark' => '',
                'ip' => '0.0.0.0',
                'platform' => $this->platformUtils->getPlatform(),
                'create_time' => time(),
                'update_time' => time(),
            );
            $inser_id = Db::table('msm_ms_coin_record')->insertGetId($coinData);
            if (!$inser_id) {
                throw new CommonException('会员转出捉米豆记录创建失败');
            }
            $feeUpdate = array(
                'coin_used' => $outCoinUsed + $outCoinUse,
            );
            Db::table('msm_member_fee')->where('mem_id', $outCollect['mem_id'])->update($feeUpdate);
            // 转入豆子
            $coinData = array(
                'category' => CoinService::CATEGORY_ADD_MOBILE,
                'out_collect_id' => $outCollect['id'],
                'out_mem_id' => $outCollect['mem_id'],
                'out_shop_id' => $outCollect['shop_id'],
                'in_collect_id' => $inCollect['id'],
                'in_mem_id' => $inCollect['mem_id'],
                'in_shop_id' => $inCollect['shop_id'],
                'coin' => $outCoinUse,
                'old_coin' => $inCoinUse,
                'order_no' => $order_no,
                'qrcode_code' => '',
                'remark' => '',
                'ip' => '0.0.0.0',
                'platform' => $this->platformUtils->getPlatform(),
                'create_time' => time(),
                'update_time' => time(),
            );
            $inser_id = Db::table('msm_ms_coin_record')->insertGetId($coinData);
            if (!$inser_id) {
                throw new CommonException('会员转入捉米豆记录创建失败');
            }
            $feeUpdate = array(
                'coin' => $inCoin + $outCoinUse,
            );
            Db::table('msm_member_fee')->where('mem_id', $inCollect['mem_id'])->update($feeUpdate);
        }
        if ($outPoint > 0) {
            // 转出积分
            $coinData = array(
                'category' => PointService::CATEGORY_REDUCE_MOBILE,
                'out_collect_id' => $outCollect['id'],
                'out_mem_id' => $outCollect['mem_id'],
                'out_shop_id' => $outCollect['shop_id'],
                'in_collect_id' => $inCollect['id'],
                'in_mem_id' => $inCollect['mem_id'],
                'in_shop_id' => $inCollect['shop_id'],
                'point' => -$outPoint,
                'old_point' => $outPoint,
                'coin' => 0,
                'order_no' => $order_no,
                'remark' => '',
                'ip' => '0.0.0.0',
                'platform' => $this->platformUtils->getPlatform(),
                'create_time' => time(),
                'update_time' => time(),
            );
            $inser_id = Db::table('msm_ms_point_record')->insertGetId($coinData);
            if (!$inser_id) {
                throw new CommonException('会员转出积分记录创建失败');
            }
            $feeUpdate = array(
                'fee_leave' => bcmul('0', '0', 2),
            );
            Db::table('msm_member_fee')->where('mem_id', $outCollect['mem_id'])->update($feeUpdate);
            // 转入积分
            $coinData = array(
                'category' => PointService::CATEGORY_ADD_MOBILE,
                'out_collect_id' => $outCollect['id'],
                'out_mem_id' => $outCollect['mem_id'],
                'out_shop_id' => $outCollect['shop_id'],
                'in_collect_id' => $inCollect['id'],
                'in_mem_id' => $inCollect['mem_id'],
                'in_shop_id' => $inCollect['shop_id'],
                'point' => $outPoint,
                'old_point' => $inPoint,
                'coin' => 0,
                'order_no' => $order_no,
                'remark' => '',
                'ip' => '0.0.0.0',
                'platform' => $this->platformUtils->getPlatform(),
                'create_time' => time(),
                'update_time' => time(),
            );
            $inser_id = Db::table('msm_ms_point_record')->insertGetId($coinData);
            if (!$inser_id) {
                throw new CommonException('会员转出积分记录创建失败');
            }
            $feeUpdate = array(
                'fee_leave' => bcadd($inPoint, $outPoint, 2),
                'fee_money' => bcadd($inFeeMoney, $outPoint, 2),
            );
            Db::table('msm_member_fee')->where('mem_id', $inCollect['mem_id'])->update($feeUpdate);
        }
        return [];
    }
}